/////////////////////////////////////////////////////////////////////
//                                                                 //
// MMPLAYER.CPP Version .95
// Base class TMMPlayer to handle multi-media input and output
//                                                                 //
//	Copyright  1995 by Kent Reisdorph, All rights reserved
//
//                                                                 //
/////////////////////////////////////////////////////////////////////

#include "mmplayer.h"

// generic constructor - just opens device if a device
// was specified
TMMPlayer::TMMPlayer(HWND hWnd, const char* device, BOOL errReporting)
{
	hWindow = hWnd;
	DeviceOpen = FALSE;
	ErrorReporting = errReporting;
	yieldProc = 0;
	if (device) ErrCode = Open(0, device);
	if (ErrCode) TMMPlayer::~TMMPlayer();
}

// one-step play constructor same effect
// as sndPlaySound(filename, SND_SYNC)
TMMPlayer::TMMPlayer(
		HWND hWnd,
		const char* filename,
		const char* device,
		DWORD flags,
		BOOL errReporting)
{
	hWindow = hWnd;
	DeviceOpen = FALSE;
	ErrorReporting = errReporting;
	yieldProc = 0;
	Open(filename, device);
	if (ErrCode) TMMPlayer::~TMMPlayer();
	Play(flags);
}

// destructor closes device if it was left open
TMMPlayer::~TMMPlayer()
{
	if (DeviceOpen) Close();
	if (yieldProc) FreeProcInstance(yieldProc);
}

//Opens a device. You can specify the filename only and mmsystem
//will determine the device to open based on the filename extension.
//Or you can specify the device only, or both.
DWORD
TMMPlayer::Open(const char* filename, const char* device)
{
  DWORD flags = 0;
	UINT devID;
	MCI_OPEN_PARMS Open;
	if (filename) flags |= MCI_OPEN_ELEMENT;
	if (device) flags |= MCI_OPEN_TYPE;
	Open.lpstrDeviceType = device;
	Open.lpstrElementName = filename;
	if (DeviceOpen) {
		devID = DeviceID;
	}
	else devID = 0;
	ErrCode = mciSendCommand(devID,
		MCI_OPEN, flags, (DWORD)(LPVOID) &Open);
	if (ErrCode && ErrorReporting) ReportError("MCI Open Error");
	if (!ErrCode) {
		DeviceOpen = TRUE;
		DeviceID = Open.wDeviceID;
	}
	return ErrCode;
}

//This Open function is primarily for AVI and animations
DWORD
TMMPlayer::Open(const char* filename,
								const char* device,
								HWND parent,
								DWORD style,
								DWORD flags)
{
	flags |= MCI_OVLY_OPEN_PARENT |
					MCI_OVLY_OPEN_WS | MCI_OPEN_ELEMENT;
	UINT devID;
	MCI_OVLY_OPEN_PARMS Open;
	if (device) flags |= MCI_OPEN_TYPE;
	Open.lpstrDeviceType = device;
	Open.lpstrElementName = filename;
	Open.dwStyle = style;
	Open.hWndParent = parent;
	if (DeviceOpen) {
		devID = DeviceID;
	}
	else devID = 0;
	ErrCode = mciSendCommand(devID,
		MCI_OPEN, flags, (DWORD)(LPVOID) &Open);
	if (ErrCode && ErrorReporting) ReportError("MCI Open Error");
	if (!ErrCode) {
		DeviceOpen = TRUE;
		DeviceID = Open.wDeviceID;
	}
	return ErrCode;
}

//Close()
//Close the device
//No flags
DWORD
TMMPlayer::Close()
{
	if (!DeviceOpen) return MCIERR_DEVICE_NOT_READY;
	ErrCode = mciSendCommand(DeviceID, MCI_CLOSE, 0, NULL);
	if (ErrCode && ErrorReporting) ReportError("MCI Close Error");
	DeviceOpen = FALSE;
	return ErrCode;
}

//Stop()
//Stop playing or recording. The difference between Stop() and Pause()
//depends on the specifics of the device driver and hardware both.
//No Flags
DWORD
TMMPlayer::Stop()
{
	if (!DeviceOpen) return MCIERR_DEVICE_NOT_READY;
	MCI_GENERIC_PARMS Generic;
	Generic.dwCallback = MAKELONG(hWindow, 0);
	ErrCode = mciSendCommand(DeviceID, MCI_STOP,	MCI_NOTIFY,
		(DWORD)(LPVOID) &Generic);
	if (ErrCode && ErrorReporting) ReportError("MCI Stop Error");
	return ErrCode;
}

// Play(flags)
// Starts playback from the beginning of the media to the end
// default flags = MM_SYNC
DWORD
TMMPlayer::Play(DWORD flags)
{
	if (!DeviceOpen) return MCIERR_DEVICE_NOT_READY;
	MCI_PLAY_PARMS Play;
	Play.dwCallback = (DWORD)hWindow;
	ErrCode = mciSendCommand(DeviceID,
		MCI_PLAY, flags, (DWORD)(LPVOID) &Play);
	if (ErrCode && ErrorReporting) ReportError("MCI Play Error");
	return ErrCode;
}

// Play(filename, flags)
// Opens the device and plays the file
// default flags = MM_SYNC
DWORD
TMMPlayer::Play(const char *filename, DWORD flags)
{
	Open(filename);
	return Play(flags);
}

// Play(from, to, flags);
// Plays from the position specefied in from to the position in to
// default flags = MM_SYNC
DWORD
TMMPlayer::Play(DWORD from, DWORD to, DWORD flags)
{
	if (!DeviceOpen) return MCIERR_DEVICE_NOT_READY;
	flags |= MCI_FROM | MCI_TO;
	if (from && !to) to = GetLength();
	MCI_PLAY_PARMS Play;
	Play.dwFrom = from;
	Play.dwTo = to;
	Play.dwCallback = MAKELONG(hWindow, 0);
	ErrCode = mciSendCommand(DeviceID,
		MCI_PLAY, flags, (DWORD)(LPVOID) &Play);
	if (ErrCode && ErrorReporting) ReportError("MCI Play Error");
	return ErrCode;
}

// PlayResource(const char* resName, HINSTANCE hInst, UINT flags)
// Plays a wave resource. The other form of this function is inline
// and allows the use of an integer resource ID.
DWORD
TMMPlayer::PlayResource(const char* resName, HINSTANCE hInst, UINT flags)
{
	flags |= SND_MEMORY;
	BOOL result;
	LPSTR lpRes;
	HANDLE hResInfo, hRes;
	hResInfo = FindResource(hInst, resName, "WAVE");
	if (hResInfo == NULL) return FALSE;
	hRes = LoadResource(hInst, (HRSRC)hResInfo);
	if (hRes == NULL) return FALSE;
	(void*)lpRes = LockResource(hRes);
	if (lpRes != NULL) {
		result = sndPlaySound(lpRes, flags);
		UnlockResource(hRes);
	}
	else result = 0;
	FreeResource(hRes);
	return result;
}

// Record(flags, from, to);
// default flags = MM_SYNC | MCI_RECORD_OVERWRITE
// if 'from' and 'to' are specified then wait until the recording
// has finished to return to the app.
DWORD
TMMPlayer::Record(DWORD from, DWORD to, DWORD flags)
{
	if (!DeviceOpen) return MCIERR_DEVICE_NOT_READY;
	flags |= MCI_TO | MCI_FROM;
	MCI_RECORD_PARMS Record;
	Record.dwFrom = from;
	Record.dwTo = to;
	Record.dwCallback = MAKELONG(hWindow, 0);
	ErrCode = mciSendCommand(DeviceID, MCI_RECORD,
					flags, (DWORD)(LPVOID)&Record);
	if (ErrCode && ErrorReporting) ReportError("MCI Record Error");
	return ErrCode;
}

// Default flags = MM_ASYNC | MCI_RECORD_OVERWRITE
// This default means that we start recording and return
// to the app. It is up to the app to Stop() the recording
// use ctrl-break to stop the recording in case of a runaway
DWORD
TMMPlayer::Record(DWORD flags)
{
	if (!DeviceOpen) return MCIERR_DEVICE_NOT_READY;
	MCI_RECORD_PARMS Record;
	Record.dwCallback = MAKELONG(hWindow, 0);
	ErrCode = mciSendCommand(DeviceID, MCI_RECORD,
					flags, (DWORD)(LPVOID)&Record);
	if (ErrCode && ErrorReporting) ReportError("MCI Record Error");
	return ErrCode;
}

// Save(filename, flags);
// Save the medium to a file
// default flags = MCI_SAVE_FILE | MCI_WAIT
DWORD
TMMPlayer::Save(const char* filename, DWORD flags)
{
	if (!DeviceOpen) return MCIERR_DEVICE_NOT_READY;
	MCI_SAVE_PARMS Save;
	Save.dwCallback = MAKELONG(hWindow, 0);
	//use the filename used when the file was opened if there is one
	if (!filename) {
		GetFileName((char*)Save.lpfilename);
		if (strlen(Save.lpfilename) < 1) return MCIERR_FILENAME_REQUIRED;
	}
	// if a filename was passed to us then use it
	if (filename) Save.lpfilename = filename;
	ErrCode = mciSendCommand(DeviceID, MCI_SAVE, flags, (DWORD)(LPVOID) &Save);
	if (ErrCode && ErrorReporting) ReportError("MCI Save Error");
	return ErrCode;
}

// Load(filename, flags);
// Load a file
// default flags = MCI_LOAD_FILE | MCI_WAIT
DWORD
TMMPlayer::Load(const char* filename, DWORD flags)
{
	if (!DeviceOpen) return MCIERR_DEVICE_NOT_READY;
	if (strlen(filename)<1) return MCIERR_DEVICE_NOT_READY;
	MCI_LOAD_PARMS Load;
	Load.dwCallback = MAKELONG(hWindow, 0);
	Load.lpfilename = filename;
	ErrCode = mciSendCommand(DeviceID, MCI_LOAD, flags, (DWORD)(LPVOID)&Load);
	if (ErrCode && ErrorReporting) ReportError("MCI Load Error");
	return ErrCode;
}

DWORD
TMMPlayer::Cue(DWORD flags)
{
	if (!DeviceOpen) return MCIERR_DEVICE_NOT_READY;
	MCI_GENERIC_PARMS Parms;
	Parms.dwCallback = MAKELONG(hWindow, 0);
	ErrCode = mciSendCommand(DeviceID, MCI_CUE,
				flags, (DWORD)(LPVOID) &Parms);
	if (ErrCode && ErrorReporting) ReportError("MCI Cue Error");
	return ErrCode;
}

// Delete(from, to);
// Deletes the medium in the buffer, not a file
// All data is deleted unless 'from' and 'to' are specified
DWORD
TMMPlayer::Delete(DWORD from, DWORD to)
{
	if (!DeviceOpen) return MCIERR_DEVICE_NOT_READY;
	DWORD flags = MCI_FROM;
	if (to) flags |= MCI_TO;
	MCI_WAVE_DELETE_PARMS Delete;
	Delete.dwCallback = MAKELONG(hWindow, 0);
	Delete.dwFrom = from;
	Delete.dwTo = to;
	ErrCode = mciSendCommand(DeviceID, MCI_DELETE,
			flags, (DWORD)(LPVOID) &Delete);
	if (ErrCode && ErrorReporting) ReportError("MCI Delete Error");
	return ErrCode;
}

// Seek(to, flags);
// Sets the current position. Subsequent actions will be performed
// from this position
// Default flags = MCI_SEEK_TO_START, this means that Seek() will
// 'rewind' the medium
DWORD
TMMPlayer::Seek(DWORD to, DWORD flags)
{
	if (!DeviceOpen) return MCIERR_DEVICE_NOT_READY;
	MCI_SEEK_PARMS Seek;
	Seek.dwCallback = MAKELONG(hWindow, 0);
	Seek.dwTo = to;
	if (to) flags = MCI_TO | MM_SYNC;
	ErrCode = mciSendCommand(DeviceID, MCI_SEEK, flags, (DWORD)(LPVOID) &Seek);
	if (ErrCode && ErrorReporting) ReportError("MCI Seek Error");
	return ErrCode;
}

// Pause(flags);
// Pauses playback or recording
// default flags = MCI_NOTIFY | MCI_WAIT
DWORD
TMMPlayer::Pause(DWORD flags)
{
	if (!DeviceOpen) return MCIERR_DEVICE_NOT_READY;
	MCI_GENERIC_PARMS Pause;
	Pause.dwCallback = MAKELONG(hWindow, 0);
	ErrCode = mciSendCommand(DeviceID, MCI_PAUSE, flags,
				(DWORD)(LPVOID) &Pause);
	if (ErrCode && ErrorReporting) ReportError("MCI Pause Error");
	return ErrCode;
}

// Resume();
// Resumes playback or recording after a Pause() command.
// no flags
DWORD
TMMPlayer::Resume()
{
	MCI_GENERIC_PARMS Resume;
	Resume.dwCallback = MAKELONG(hWindow, 0);
	ErrCode = mciSendCommand(DeviceID, MCI_RESUME, MCI_NOTIFY,
				(DWORD)(LPVOID) &Resume);
	if (ErrCode && ErrorReporting) ReportError("MCI Resume Error");
	return ErrCode;
}

// GetStatusItem(item);
// Returns the status of the item requested or an error code
// in the event of an error.
// This function is used by many inline member functions to
// obtain status information about a device
DWORD
TMMPlayer::GetStatusItem(DWORD item)
{
	if (!DeviceOpen) return MCIERR_DEVICE_NOT_READY;
	MCI_STATUS_PARMS Status;
	//Status.dwCallback = MAKELONG(hWindow, 0);
	Status.dwItem = item;
	ErrCode = mciSendCommand(DeviceID, MCI_STATUS, MCI_STATUS_ITEM,
				(DWORD)(LPVOID) &Status);
	if (ErrCode && ErrorReporting) ReportError("MCI Status Error");
	if (ErrCode) return ErrCode;
	return Status.dwReturn;
}

// GetDeviceCaps(item);
// Returns the capability of a device requested in 'item'
// or an error code in the event of an error.
// This function is used by some inline member functions to
// obtain device capability information about a device
DWORD
TMMPlayer::GetDeviceCaps(DWORD item)
{
	if (!DeviceOpen) return MCIERR_DEVICE_NOT_READY;
	MCI_GETDEVCAPS_PARMS Caps;
	//Status.dwCallback = MAKELONG(hWindow, 0);
	Caps.dwItem = item;
	ErrCode = mciSendCommand(DeviceID, MCI_GETDEVCAPS,
				MCI_GETDEVCAPS_ITEM, (DWORD)(LPVOID) &Caps);
	if (ErrCode && ErrorReporting) ReportError("MCI Error");
	if (ErrCode) return ErrCode;
	else return Caps.dwReturn;
}

// GetDeviceName(name);
// 'name' is set to the device name upon success
DWORD
TMMPlayer::GetDeviceName(char* name)
{
	if (!DeviceOpen) return MCIERR_DEVICE_NOT_READY;
	MCI_SYSINFO_PARMS Info;
	Info.dwRetSize = 255;
	Info.lpstrReturn = name;
	ErrCode = mciSendCommand(DeviceID, MCI_SYSINFO,
				MCI_SYSINFO_INSTALLNAME, (DWORD)(LPVOID) &Info);
	if (ErrCode && ErrorReporting) ReportError("MCI Error");
	return ErrCode;
}

// GetInfoItem(item, info);
// Upon success 'info' contains textual information about a device
// requested in 'item'
DWORD
TMMPlayer::GetInfoItem(DWORD item, char* info)
{
	MCI_INFO_PARMS Info;
	Info.dwRetSize = 255;
	Info.lpstrReturn = info;
	ErrCode = mciSendCommand(DeviceID, MCI_INFO, item,
				(DWORD)(LPVOID) &Info);
	if (ErrCode && ErrorReporting) ReportError("MCI Error");
	return ErrCode;
}

// Set(flags, options)
// Sets device specific parameters
// This function is used by inline member functions
DWORD
TMMPlayer::Set(DWORD flags, DWORD options)
{
	MCI_SET_PARMS Set;
	if (flags | MCI_SET_AUDIO) Set.dwAudio = options;
	if (flags | MCI_SET_TIME_FORMAT) Set.dwTimeFormat = options;
	ErrCode = mciSendCommand(DeviceID, MCI_SET, flags,
				(DWORD)(LPVOID) &Set);
	if (ErrCode && ErrorReporting) ReportError("MCI Error");
	return ErrCode;
}

// SetVolume(volume);
// Sets both the right and left channels to 'volume'
// If a device is mono then the high order word is ignored
// 'volume' represents a volume percentage
// SetVolume(0) is off, SetVolume(100) is full volume
// This function checks to see whether the device is a sequencer
// or 'other' and calls the appropriate function to set volume
// This function could use some more work.
DWORD
TMMPlayer::SetVolume(int volume)
{
	DWORD devType = GetDeviceType();
	DWORD vol = MAKELONG(((volume*65535L)/100), ((volume*65535L)/100));
	switch(devType) {
		case MCI_DEVTYPE_SEQUENCER : {
			MIDIOUTCAPS midiCaps;
			midiOutGetDevCaps(0, &midiCaps, sizeof(midiCaps));
			if (midiCaps.dwSupport & MIDICAPS_VOLUME)
				ErrCode = midiOutSetVolume(0, vol);
			break;
			}
		case MCI_DEVTYPE_DIGITAL_VIDEO :
		case MCI_DEVTYPE_WAVEFORM_AUDIO : {
			WAVEOUTCAPS waveCaps;
			waveOutGetDevCaps(0, &waveCaps, sizeof(waveCaps));
			if (waveCaps.dwSupport & WAVECAPS_VOLUME)
			ErrCode = waveOutSetVolume(0, vol);
			break;
		}
		case MCI_DEVTYPE_CD_AUDIO : {
			AUXCAPS auxCaps;
			auxGetDevCaps(0, &auxCaps, sizeof(auxCaps));
			if (auxCaps.dwSupport & AUXCAPS_VOLUME)
			ErrCode = auxSetVolume(0, vol);
			break;
		}
	}
	if ( ErrCode ) ReportError();
	return ErrCode;
}

// SetVolume(lvol, rvol);
// Sets the right and left channels independently
// If a device is mono then the high order word is ignored
// lvol and rvol represent volume percentages
// SetVolume(0,0) is off, SetVolume(100,100) is full volume
// This function is not completed. It should check to see what
// type of device is open as the above function does.
DWORD
TMMPlayer::SetVolume(int lvol, int rvol)
{
	DWORD vol = MAKELONG(((rvol*65535L)/100), ((lvol*65535L)/100));
	ErrCode = waveOutSetVolume(0, vol);
	if (ErrCode && ErrorReporting) ReportError("MCI Error");
	if (ErrCode) return ErrCode;
	return 0;
}

// SetYieldProcData(DWORD data)
// Supposed to change the data sent to the yield proc in
// SetYieldProc but it doesn't work.
void
TMMPlayer::SetYieldProcData(DWORD data)
{
	mciGetYieldProc(DeviceID, (LPDWORD)data);
}

// SetYieldProc(BOOL enabled, DWORD data, BOOL msgLoop)
// Sets up the yield process.
void
TMMPlayer::SetYieldProc(BOOL enabled, DWORD data, BOOL msgLoop)
{
	if (!enabled) {
		mciSetYieldProc(DeviceID, NULL, 0);
		FreeProcInstance(yieldProc);
		yieldProc = 0;
		return;
	}
	if (!yieldProc) yieldProc = MakeProcInstance((FARPROC)YieldProc,
				(HINSTANCE)GetWindowWord(hWindow, GWW_HINSTANCE));
	ypStruct.classInst = (DWORD)this;
	ypStruct.dwData = data;
	ypStruct.msgLoop = msgLoop;
	mciSetYieldProc(DeviceID, (YIELDPROC)yieldProc,
		(DWORD)(LPVOID)&ypStruct);
}

// This is the callback function. It retrieves the class instance
// from 'data' and then calls the class member YieldProcess to do
// the actual processing. If messageLoop is TRUE in SetYielProc
// then it enables the message loop.
int CALLBACK _export
TMMPlayer::YieldProc(UINT devID, DWORD data)
{
	MSG msg;
	YIELD_PROC_STRUCT *ypStruct = (YIELD_PROC_STRUCT*) data;
	TMMPlayer* instPtr = (TMMPlayer*)ypStruct->classInst;
	if (ypStruct->msgLoop) {
		if (GetMessage (&msg, NULL, 0, 0))
		{
			TranslateMessage (&msg) ;
			DispatchMessage (&msg) ;
		}
	}
	return instPtr->YieldProcess(devID, ypStruct->dwData);
}

// GetVolume();
// Returns the current volume of the wave output device
// This function is not completed at this time as it should
// check to see if the device is wave, midi, or aux
DWORD
TMMPlayer::GetVolume()
{
	DWORD vol;
	ErrCode = waveOutGetVolume(0, &vol);
	if (ErrCode && ErrorReporting) ReportError("MCI Error");
	return vol;
}

// ReportError(title);
// Displays a message box with the MCI error if ErrorReporting
// is set to TRUE
void
TMMPlayer::ReportError(const char* title)
{
	if (ErrCode == MCIERR_DEVICE_OPEN) {
		mciSendCommand(DeviceID, MCI_CLOSE, 0, NULL);
		return;
	}
	mciGetErrorString(ErrCode, ErrMsg, sizeof(ErrMsg));
	MessageBeep(MB_ICONEXCLAMATION);
	MessageBox(hWindow, ErrMsg, title, MB_ICONEXCLAMATION);
}

// The following functions don't work as I expect they should so
// they have been set aside for the time being.
DWORD
TMMPlayer::Cut(DWORD flags)
{
	if (!DeviceOpen) return MCIERR_DEVICE_NOT_READY;
	MCI_GENERIC_PARMS Generic;
	Generic.dwCallback = MAKELONG(hWindow, 0);
	ErrCode = mciSendCommand(DeviceID, MCI_CUT, flags,
				(DWORD)(LPVOID) &Generic);
	if (ErrCode && ErrorReporting) ReportError("MCI Clipboard Cut Error");
	return ErrCode;
}

DWORD
TMMPlayer::Copy(DWORD flags)
{
	if (!DeviceOpen) return MCIERR_DEVICE_NOT_READY;
	MCI_GENERIC_PARMS Generic;
	Generic.dwCallback = MAKELONG(hWindow, 0);
	ErrCode = mciSendCommand(DeviceID, MCI_COPY, flags,
				(DWORD)(LPVOID) &Generic);
	if (ErrCode && ErrorReporting) ReportError("MCI Clipboard Copy Error");
	return ErrCode;
}

DWORD
TMMPlayer::Paste(DWORD flags)
{
	if (!DeviceOpen) return MCIERR_DEVICE_NOT_READY;
	MCI_GENERIC_PARMS Generic;
	Generic.dwCallback = MAKELONG(hWindow, 0);
	ErrCode = mciSendCommand(DeviceID, MCI_PASTE, flags,
				(DWORD)(LPVOID) &Generic);
	if (ErrCode && ErrorReporting) ReportError("MCI Clipboard Paste Error");
	return ErrCode;
}



